<?php
/**
 * Image tools: crop 3:4 (kalau perlu) + remove background (opsional)
 * Mendukung 3 jalur remove-bg:
 *  - remove.bg API (jika ada API key)
 *  - rembg HTTP service (jika ada URL)
 *  - rembg CLI lokal (jika tersedia di server)
 */

if (!function_exists('ensure_dir')) {
  function ensure_dir(string $path): void {
    $dir = is_dir($path) ? $path : dirname($path);
    if (!is_dir($dir)) {
      @mkdir($dir, 0775, true);
    }
  }
}

/**
 * Simple copy/crop placeholder.
 * Catatan: foto sudah dicrop 3:4 di sisi browser, jadi di server cukup salin.
 * Kalau tetap mau jaga-jaga, fungsi ini hanya memastikan file tersalin ke $outPath.
 */
function crop_to_ratio(string $srcPath, string $outPath, int $wRatio = 3, int $hRatio = 4): bool {
  ensure_dir($outPath);
  return @copy($srcPath, $outPath);
}

/** Wrapper lama, tetap ada untuk kompatibilitas */
function remove_bg_if_requested(string $inPath, string $outPath, bool $doRemove, array $cfg = []): bool {
  if (!$doRemove) {
    ensure_dir($outPath);
    return @copy($inPath, $outPath);
  }
  return remove_bg_best($inPath, $outPath, $cfg);
}

/**
 * Jalur terbaik remove background yang tersedia.
 * Urutan: remove.bg -> rembg HTTP -> rembg CLI -> fallback copy
 */
function remove_bg_best(string $inPath, string $outPath, array $cfg = []): bool {
  // 1) remove.bg
  $key = trim($cfg['removebg_api_key'] ?? '');
  if ($key !== '') {
    $ok = remove_bg_via_removebg($inPath, $outPath, $key);
    if ($ok) return true;
  }

  // 2) rembg HTTP service
  $rembgUrl = trim($cfg['rembg']['url'] ?? '');
  if ($rembgUrl !== '') {
    $ok = remove_bg_via_rembg_http($inPath, $outPath, $rembgUrl, (int)($cfg['rembg']['timeout'] ?? 40), (array)($cfg['rembg']['headers'] ?? []));
    if ($ok) return true;
  }

  // 3) rembg CLI lokal (kalau ada)
  if (function_exists('shell_exec')) {
    $bin = trim((string)@shell_exec('command -v rembg'));
    if ($bin !== '') {
      $ok = remove_bg_via_rembg_cli($inPath, $outPath, $bin);
      if ($ok) return true;
    }
  }

  // Fallback: tetap pakai foto asli
  ensure_dir($outPath);
  return @copy($inPath, $outPath);
}

function remove_bg_via_removebg(string $inPath, string $outPath, string $apiKey): bool {
  if (!function_exists('curl_init')) return false;

  $ch = curl_init('https://api.remove.bg/v1.0/removebg');
  $post = [
    'image_file' => new CURLFile($inPath),
    'size'       => 'auto',
    'format'     => 'png',
  ];
  curl_setopt_array($ch, [
    CURLOPT_POST           => true,
    CURLOPT_POSTFIELDS     => $post,
    CURLOPT_HTTPHEADER     => ['X-Api-Key: ' . $apiKey],
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT        => 45,
  ]);
  $raw = curl_exec($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);

  if ($code === 200 && $raw) {
    ensure_dir($outPath);
    return (bool)@file_put_contents($outPath, $raw);
  }
  return false;
}

function remove_bg_via_rembg_http(string $inPath, string $outPath, string $url, int $timeout = 40, array $headers = []): bool {
  if (!function_exists('curl_init')) return false;
  $img = @file_get_contents($inPath);
  if ($img === false) return false;

  // 2 percobaan: (a) kirim raw bytes, (b) kirim multipart "image"
  $trials = [
    [
      'headers' => array_merge(['Content-Type: application/octet-stream'], array_map(fn($k,$v)=>$k.': '.$v, array_keys($headers), $headers)),
      'fields'  => $img,
    ],
    [
      'headers' => array_values(array_map(fn($k,$v)=>$k.': '.$v, array_keys($headers), $headers)),
      'fields'  => ['image' => new CURLFile($inPath)],
    ],
  ];

  foreach ($trials as $t) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
      CURLOPT_POST           => true,
      CURLOPT_POSTFIELDS     => $t['fields'],
      CURLOPT_HTTPHEADER     => $t['headers'],
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_TIMEOUT        => max(10, $timeout),
    ]);
    $raw  = curl_exec($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $type = (string)curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
    curl_close($ch);

    if ($code === 200 && $raw && str_contains(strtolower($type), 'image')) {
      ensure_dir($outPath);
      return (bool)@file_put_contents($outPath, $raw);
    }
  }
  return false;
}

function remove_bg_via_rembg_cli(string $inPath, string $outPath, string $bin = 'rembg'): bool {
  // rembg i <in> <out>
  ensure_dir($outPath);
  $cmd = escapeshellcmd($bin) . ' i ' . escapeshellarg($inPath) . ' ' . escapeshellarg($outPath) . ' 2>&1';
  @exec($cmd, $o, $ret);
  return ($ret === 0) && is_file($outPath) && filesize($outPath) > 0;
}
